/*
 * Decompiled with CFR 0.152.
 */
package org.gradle.internal.file.impl;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.LongSupplier;
import java.util.function.Predicate;
import org.gradle.internal.file.Deleter;
import org.jspecify.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultDeleter
implements Deleter {
    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultDeleter.class);
    private final LongSupplier timeProvider;
    private final Predicate<? super File> isSymlink;
    private final boolean runGcOnFailedDelete;
    private static final int DELETE_RETRY_SLEEP_MILLIS = 10;
    @VisibleForTesting
    static final int MAX_REPORTED_PATHS = 16;
    @VisibleForTesting
    static final int EMPTY_DIRECTORY_DELETION_ATTEMPTS = 10;
    @VisibleForTesting
    static final String HELP_FAILED_DELETE_CHILDREN = "Failed to delete some children. This might happen because a process has files open or has its working directory set in the target directory.";
    @VisibleForTesting
    static final String HELP_NEW_CHILDREN = "New files were found. This might happen because a process is still writing to the target directory.";

    public DefaultDeleter(LongSupplier timeProvider, Predicate<? super File> isSymlink, boolean runGcOnFailedDelete) {
        this.timeProvider = timeProvider;
        this.isSymlink = isSymlink;
        this.runGcOnFailedDelete = runGcOnFailedDelete;
    }

    @Override
    public boolean deleteRecursively(File target) throws IOException {
        return this.deleteRecursively(target, false);
    }

    @Override
    public boolean deleteRecursively(File root, boolean followSymlinks) throws IOException {
        if (root.exists()) {
            return this.deleteRecursively(root, followSymlinks ? Handling.FOLLOW_SYMLINKED_DIRECTORIES : Handling.DO_NOT_FOLLOW_SYMLINKS);
        }
        return false;
    }

    @Override
    public boolean ensureEmptyDirectory(File target) throws IOException {
        return this.ensureEmptyDirectory(target, false);
    }

    @Override
    public boolean ensureEmptyDirectory(File root, boolean followSymlinks) throws IOException {
        if (root.exists()) {
            if (root.isDirectory() && (followSymlinks || !this.isSymlink.test(root))) {
                return this.deleteRecursively(root, followSymlinks ? Handling.KEEP_AND_FOLLOW_SYMLINKED_DIRECTORIES : Handling.KEEP_AND_DO_NOT_FOLLOW_CHILD_SYMLINKS);
            }
            this.tryHardToDeleteOrThrow(root);
        }
        if (!root.mkdirs()) {
            throw new IOException("Couldn't create directory: " + root);
        }
        return true;
    }

    @Override
    public boolean delete(File target) throws IOException {
        if (!target.exists()) {
            return false;
        }
        this.tryHardToDeleteOrThrow(target);
        return true;
    }

    private boolean deleteRecursively(File root, Handling handling) throws IOException {
        LOGGER.debug("Deleting {}", (Object)root);
        long startTime = this.timeProvider.getAsLong();
        LinkedHashMap<String, FileDeletionResult> failedPaths = new LinkedHashMap<String, FileDeletionResult>();
        boolean attemptedToRemoveAnything = this.deleteRecursively(startTime, root, root, handling, failedPaths);
        if (!failedPaths.isEmpty()) {
            this.throwWithHelpMessage(startTime, root, handling, failedPaths, false);
        }
        return attemptedToRemoveAnything;
    }

    private boolean deleteRecursively(long startTime, File baseDir, File file, Handling handling, Map<String, FileDeletionResult> failedPaths) throws IOException {
        FileDeletionResult result;
        if (this.shouldRemoveContentsOf(file, handling)) {
            File[] contents = file.listFiles();
            if (contents == null) {
                return false;
            }
            boolean attemptedToDeleteAnything = false;
            for (File item : contents) {
                this.deleteRecursively(startTime, baseDir, item, handling.getDescendantHandling(), failedPaths);
                attemptedToDeleteAnything = true;
            }
            if (handling.shouldKeepEntry()) {
                return attemptedToDeleteAnything;
            }
        }
        if (!(result = this.tryHardToDelete(file)).isSuccessful) {
            failedPaths.put(file.getAbsolutePath(), result);
            if (failedPaths.size() == 16) {
                this.throwWithHelpMessage(startTime, baseDir, handling, failedPaths, true);
            }
        }
        return true;
    }

    private boolean shouldRemoveContentsOf(File file, Handling handling) {
        return file.isDirectory() && (handling.shouldFollowLinkedDirectory() || !this.isSymlink.test(file));
    }

    private void tryHardToDeleteOrThrow(File file) throws IOException {
        FileDeletionResult result = this.tryHardToDelete(file);
        if (!result.isSuccessful) {
            throw new IOException("Couldn't delete " + file, result.exception);
        }
    }

    private FileDeletionResult tryHardToDelete(File file) {
        FileDeletionResult lastResult = this.deleteFile(file);
        if (lastResult.isSuccessful) {
            return lastResult;
        }
        if (this.runGcOnFailedDelete) {
            System.gc();
        }
        for (int failedAttempts = 1; failedAttempts < 10; ++failedAttempts) {
            try {
                Thread.sleep(10L);
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
            }
            lastResult = this.deleteFile(file);
            if (!lastResult.isSuccessful) continue;
            return lastResult;
        }
        return lastResult;
    }

    protected FileDeletionResult deleteFile(File file) {
        try {
            return FileDeletionResult.withoutException(Files.deleteIfExists(file.toPath()));
        }
        catch (IOException original) {
            if (file.setWritable(true)) {
                try {
                    return FileDeletionResult.withoutException(Files.deleteIfExists(file.toPath()));
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
            return FileDeletionResult.withException(original);
        }
    }

    private void throwWithHelpMessage(long startTime, File file, Handling handling, Map<String, FileDeletionResult> failedPaths, boolean more) throws IOException {
        IOException ex = new IOException(this.buildHelpMessageForFailedDelete(startTime, file, handling, failedPaths.keySet(), more));
        for (FileDeletionResult result : failedPaths.values()) {
            if (result.exception == null) continue;
            ex.addSuppressed(result.exception);
        }
        throw ex;
    }

    private String buildHelpMessageForFailedDelete(long startTime, File file, Handling handling, Collection<String> failedPaths, boolean more) {
        StringBuilder help = new StringBuilder("Unable to delete ");
        if (this.isSymlink.test(file)) {
            help.append("symlink to ");
        }
        if (file.isDirectory()) {
            help.append("directory ");
        } else {
            help.append("file ");
        }
        help.append('\'').append(file).append('\'');
        if (this.shouldRemoveContentsOf(file, handling)) {
            Collection<String> newPaths;
            String absolutePath = file.getAbsolutePath();
            failedPaths.remove(absolutePath);
            if (!failedPaths.isEmpty()) {
                help.append("\n  ").append(HELP_FAILED_DELETE_CHILDREN);
                for (String failed : failedPaths) {
                    help.append("\n  - ").append(failed);
                }
                if (more) {
                    help.append("\n  - and more ...");
                }
            }
            if (!(newPaths = DefaultDeleter.listNewPaths(startTime, file, failedPaths)).isEmpty()) {
                help.append("\n  ").append(HELP_NEW_CHILDREN);
                for (String newPath : newPaths) {
                    help.append("\n  - ").append(newPath);
                }
                if (newPaths.size() == 16) {
                    help.append("\n  - and more ...");
                }
            }
        }
        return help.toString();
    }

    private static Collection<String> listNewPaths(long startTime, File directory, Collection<String> failedPaths) {
        ArrayList<String> paths = new ArrayList<String>(16);
        ArrayDeque<File> stack = new ArrayDeque<File>();
        stack.push(directory);
        while (!stack.isEmpty() && paths.size() < 16) {
            File[] children;
            File current = (File)stack.pop();
            String absolutePath = current.getAbsolutePath();
            if (!current.equals(directory) && !failedPaths.contains(absolutePath) && current.lastModified() >= startTime) {
                paths.add(absolutePath);
            }
            if (!current.isDirectory() || (children = current.listFiles()) == null) continue;
            for (File child : children) {
                stack.push(child);
            }
        }
        return paths;
    }

    private static enum Handling {
        KEEP_AND_FOLLOW_SYMLINKED_DIRECTORIES(true, true){

            @Override
            public Handling getDescendantHandling() {
                return FOLLOW_SYMLINKED_DIRECTORIES;
            }
        }
        ,
        KEEP_AND_DO_NOT_FOLLOW_CHILD_SYMLINKS(true, true){

            @Override
            public Handling getDescendantHandling() {
                return DO_NOT_FOLLOW_SYMLINKS;
            }
        }
        ,
        FOLLOW_SYMLINKED_DIRECTORIES(false, true){

            @Override
            public Handling getDescendantHandling() {
                return FOLLOW_SYMLINKED_DIRECTORIES;
            }
        }
        ,
        DO_NOT_FOLLOW_SYMLINKS(false, false){

            @Override
            public Handling getDescendantHandling() {
                return DO_NOT_FOLLOW_SYMLINKS;
            }
        };

        private final boolean shouldKeepEntry;
        private final boolean shouldFollowLinkedDirectory;

        private Handling(boolean shouldKeepEntry, boolean shouldFollowLinkedDirectory) {
            this.shouldKeepEntry = shouldKeepEntry;
            this.shouldFollowLinkedDirectory = shouldFollowLinkedDirectory;
        }

        public boolean shouldKeepEntry() {
            return this.shouldKeepEntry;
        }

        public boolean shouldFollowLinkedDirectory() {
            return this.shouldFollowLinkedDirectory;
        }

        public abstract Handling getDescendantHandling();
    }

    protected static final class FileDeletionResult {
        private final boolean isSuccessful;
        private final @Nullable Exception exception;

        static FileDeletionResult withoutException(boolean isSuccessful) {
            return new FileDeletionResult(isSuccessful, null);
        }

        static FileDeletionResult withException(Exception exception) {
            return new FileDeletionResult(false, exception);
        }

        private FileDeletionResult(boolean isSuccessful, @Nullable Exception exception) {
            this.isSuccessful = isSuccessful;
            this.exception = exception;
        }
    }
}

